/******************************************************************************
 * Archive.h - Archive                                                        *
 ******************************************************************************
 * Projet       : Archive(R)                                                  *
 * Auteur       : Arnaud Storq (http://norecess.planet-d.net)                 *
 ******************************************************************************/
#ifndef __ARCHIVE_H__
#define __ARCHIVE_H__

// ---------------------------------------------------------------------------- CODE RETOURS
// A chaque appel d'une fonction, une variable du type ArchiveResultat est retourne.
// Cela permet de savoir si l'operation s'est correctement deroule !
enum ArchiveResultat
{
   ID_ARCHIVE_OK,                               // Aucune erreur

   ID_ARCHIVE_ERREUR_MAUVAISE_VERSION,          // Archive invalide
   
   ID_ARCHIVE_ERREUR_FICHIER_DEJA_EXISTANT,     // Fichier deja existant lors de l'ajout
   ID_ARCHIVE_ERREUR_FICHIER_INEXISTANT         // Fichier inexistant lors de la suppression
};

// ---------------------------------------------------------------------------- TYPE COMPRESSION
// Le type de compression utilis. Certes dans notre programme nous n'utilisons
// actuellemt que la version compresse avec la ZLib, mais qu'importe : nous serons
// capable de rajouter un autre type de compression (codec) plus tard par exemple..
// Cela ameliorer grandement l'evolutivit de notre programme et de notre format 
// de fichier d'archivage..
enum TypeCompression
{
   ID_ARCHIVE_COMPRESSION_AUCUNE,
   ID_ARCHIVE_COMPRESSION_ZLIB
};

// ---------------------------------------------------------------------------- STRUCTURES
// L'entete se situe toujours au debut du fichier d'archivage.
// La signature permet de connaitre le numro de version du fichier d'archivage,
// cela sera par exemple utile pour une future version..
struct ArchiveEntete
{
    char signature[4];
    int nombreFichiers;
};
    
// Aprs l'entete se situe la description du 1er fichier.
// On y trouve des informations essentielles : le nom du fichier, sa taille,
// le type de compression utilise et enfin sa taille compresse.
// Le pointeur est egalement stocke dans l'archive, cela nous simplifie le
// travail au sein de l'outil. Ce n'est pas grave si une valeur invalide est
// stocke dans l'archive pour cette valeur..
struct ArchiveElement
{
    char nomFichier[256];
    int tailleOriginale;

    TypeCompression typeCompression;
    int tailleCompressee;
        
    char *pointeur;
};

// ---------------------------------------------------------------------------- ARCHIVE
/** Represente le gestionnaire d'archive. */
class Archive 
{
public:
    Archive();
    ~Archive();

public:
    // Vider l'archive consiste  effacer tout le contenu de l'archive.
    // Cette fonction est notemment appelle quand il faut ouvrir une
    // archive alors qu'un autre archive est dj ouverte, ou dans le
    // destructeur pour quitter correctement la classe sans "fuites" de mmoire.
    ArchiveResultat archiveVider();

public:
    // Ouvre l'archive en specifiant son nom de fichier.
    // Cela place tous les fichiers de l'archive en mmoire sous forme compresse.
    ArchiveResultat archiveOuvrir(string &nomFichier);
    
    // Sauvegarde le contenu de l'archive (qui est actuellement stocke en mmoire)
    // dans le fichier specifi par son nom en parametres.
    ArchiveResultat archiveSauvegarder(string &nomFichier);

public:
    // Ajoute un fichier  l'archive (si il est inexistant).
    // Le tableau pass en parametres est alors compress avec la Zlib. 
    // Un element de type ArchiveElement est rempli et ajout au tableau d'elements (vecteur).
    ArchiveResultat fichierAjouter(string &nomFichier, char *pointeur, int taille);
    
    // Supprime un fichier de l'archive (si il existe).
    // L'element ArchiveElement le correspondant est alors retir du tableau d'elements (vecteur).
    ArchiveResultat fichierSupprimer(string &nomFichier);
    
private:
    // Compresse l'element pass en parametres en utilisant la Zlib.
    // Cette mthode est place en prive car le developpeur n'a pas
    //  influencer le traitement de compression.
    ArchiveResultat fichierCompresser(ArchiveElement &element);

public:
    // Decompresse l'element pass en parametres en utilisant la Zlib.
    // pointeur et taille sont tous les deux allous par la mthode,
    // c'est au developpeur de les desallouer par la suite.
    ArchiveResultat fichierDecompresser(string &nomFichier, char *&pointeur, int &taille);
    
public:
    // Retourne le nombre de fichiers stocks dans l'archive.
    int archiveObtenirNombreFichiers();
    
    // Retourne le nom du fichier identifi par son indice (entre 0 et NombreFichiers).    
    string archiveObtenirNomFichierParIndice(int indice);
        
    // Cette fonction permet d'obtenir l'element specifi par son nom de fichier.
    // Cela permet dans l'interface de pouvoir afficher la taille compresse,
    // la taille originale ou encore connaitre son type de compression.
    ArchiveElement fichierObtenirElement(string &nomFichier);
    
private:
    // Un vecteur STL permet la manipulation d'un tableau avec le type specifi
    // (ici, ArchiveElement) (c'est un template, mais rien de compliqu pour son
    // utilisation). Cela nous permet d'implementer la gestion d'un tableau
    // avec ses ajouts, suppressions, etc.
    vector<ArchiveElement> m_elements;
    
    // Cet iterateur STL va nous etre utilie pour "traverser" notre vecteur.
    vector<ArchiveElement>::iterator m_iterateur;
};

// ---------------------------------------------------------------------------- FIN
#endif // __ARCHIVE_H__

